script_name("Wiki Viewer")
script_author("Pauly_Galtieri")

require 'lib.moonloader'
local skinData = {}
local houseData = {}
local accessoryData = {}
local pendingVehicleUpdates = {}
local pendingVehicleRequests = {}
local imgui = require 'imgui'
local requests = require 'requests'
local json = require 'json'

local mainWindowOpen = imgui.ImBool(false)
local menuItem = nil
local searchQuery = imgui.ImBuffer(64)

local transportData = {}
local vehicleDetails = {}
local loading = false
local maxOutputLength = 8000 -- ограничение на вывод текста


function main()
	
    while not isSampAvailable() do wait(100) end
    sampRegisterChatCommand("wiki", toggleWindow)
    sampAddChatMessage("/wiki loaded!", 0x00FF00)
	
	imgui.ShowCursor = true
	imgui.Blocking = false

    while true do
        wait(0)

       -- безопасное обновление vehicleDetails
		if #pendingVehicleUpdates > 0 then
			for i = 1, #pendingVehicleUpdates do
				local update = pendingVehicleUpdates[i]
				vehicleDetails[update.id] = {
					Class = update.data.Class or "-",
					Speed = update.data.Speed or "-",
					Drive = update.data.Drive or "-"
				}
			end
			pendingVehicleUpdates = {}
		end

		-- по одному запросу за кадр
		if #pendingVehicleRequests > 0 then
			local id = table.remove(pendingVehicleRequests, 1)
			fetchVehicleInfo(id)
		end

        imgui.Process = mainWindowOpen.v
    end
end

function isDialogOrChatActive()
    return isSampfuncsConsoleActive() or sampIsChatInputActive() or sampIsDialogActive()
end

function toggleWindow()
    mainWindowOpen.v = not mainWindowOpen.v
end

function fetchSkinData()
    if loading then return end
    lua_thread.create(function()
        loading = true
        skinData = {}

        local id = tonumber(searchQuery.v:match("^%s*(.-)%s*$")) or 0
        local url = "https://s1k.ru/evolve/skin_info.php?id=" .. tostring(id)

        local res = requests.get(url)
        if res.status_code == 200 then
            local ok, data = pcall(json.decode, res.text)
            if ok and type(data) == "table" then
                skinData = {
                    Name   = data.Name   or "-",
                    Price  = data.Price  or "-",
                    Model  = data.Model  or "-",
                    Gender = data.Gender or "-"
                }
            else
                skinData = { Name = "Ошибка JSON" }
            end
        else
            skinData = { Name = "HTTP " .. res.status_code }
        end

        loading = false
    end)
end

function fetchHouseData()
    if loading then return end
    lua_thread.create(function()
        loading = true
        houseData = {}

        local id = tonumber(searchQuery.v:match("^%s*(.-)%s*$")) or 0
        local url = "https://s1k.ru/evolve/house.php?id=" .. tostring(id)

        local res = requests.get(url)
        if res.status_code == 200 then
            local ok, data = pcall(json.decode, res.text)
            if ok and type(data) == "table" then
                houseData = {
                    Class  = data.Class  or "-",
                    Zone   = data.Zone   or "-",
                    Garage = data.Garage or "-"
                }
            else
                houseData = { Class = "Ошибка JSON" }
            end
        else
            houseData = { Class = "HTTP "..res.status_code }
        end

        loading = false
    end)
end

function fetchAccessoryData()
    if loading then return end
    lua_thread.create(function()
        loading = true
        accessoryData = {}

        local url = "https://s1k.ru/evolve/accessories.php"
        local query = searchQuery.v:match("^%s*(.-)%s*$")
        if query ~= "" then
            url = url .. "?search=" .. urlEncode(query)
        end

        local res = requests.get(url)
        if res.status_code == 200 then
            local ok, data = pcall(json.decode, res.text)
            if ok and type(data) == "table" then
                accessoryData = data
            else
                accessoryData = {{ ID = 0, Alternative = "Ошибка JSON" }}
            end
        else
            accessoryData = {{ ID = 0, Alternative = "Ошибка HTTP: " .. res.status_code }}
        end

        loading = false
    end)
end

function fetchVehicleInfo(id)
    lua_thread.create(function()
        local url = "https://s1k.ru/evolve/vehicle_info.php?ID=" .. tostring(id)
        local res = requests.get(url)
        if res.status_code == 200 then
            local ok, data = pcall(json.decode, res.text)
            if ok and type(data) == "table" then
                table.insert(pendingVehicleUpdates, {id = id, data = data})
            end
        end
    end)
end

function fetchTransportData()
    if loading then return end -- 🔒 не даём запустить повторно
    lua_thread.create(function()
        loading = true
        transportData = {}
        vehicleDetails = {}

        local url = "https://s1k.ru/evolve/vehicles.php"
        local query = searchQuery.v:match("^%s*(.-)%s*$")
        if query ~= "" then
            url = url .. "?search=" .. urlEncode(query)
        end

        local res = requests.get(url)
        if res.status_code == 200 then
            local ok, data = pcall(json.decode, res.text)
            if ok and type(data) == "table" then
                transportData = data
                for i = 1, #data do
					local id = tonumber(data[i].ID)
					if id then table.insert(pendingVehicleRequests, id) end
				end
            else
                transportData = {{ ID = 0, Name = "Ошибка разбора JSON" }}
            end
        else
            transportData = {{ ID = 0, Name = "Ошибка HTTP: " .. res.status_code }}
        end
        loading = false
    end)
end


function urlEncode(str)
    if str == nil then return "" end
    str = str:gsub("\n", "\r\n")
    str = str:gsub("([^%w%-_%.%~])", function(c)
        return string.format("%%%02X", string.byte(c))
    end)
    return str
end

function imgui.OnDrawFrame()
    if not mainWindowOpen.v then return end

    imgui.SetNextWindowSize(imgui.ImVec2(700, 500), imgui.Cond.FirstUseEver)

    if not imgui.Begin("Evolve Wiki by Pauly_Galtieri for SFN", mainWindowOpen, imgui.WindowFlags.NoCollapse) then
        imgui.End()
        return
    end

    local blockInput = isDialogOrChatActive()
    if loading then imgui.BeginDisabled() end

    imgui.Columns(2, "##cols", false)
    imgui.SetColumnWidth(0, 150)

    imgui.BeginChild("MenuPane", imgui.ImVec2(0, 0), false)
        if imgui.Selectable("Транспорт",  menuItem == "transport")   then menuItem = "transport"   end
        if imgui.Selectable("Аксессуары", menuItem == "accessories") then menuItem = "accessories" end
        if imgui.Selectable("Дома",       menuItem == "houses")      then menuItem = "houses"      end
        if imgui.Selectable("Скины",      menuItem == "skins")       then menuItem = "skins"       end
    imgui.EndChild()

    imgui.NextColumn()

    imgui.PushItemWidth(-100)
    if imgui.InputText("Поиск", searchQuery, imgui.InputTextFlags.EnterReturnsTrue) then
        if     menuItem == "transport"   then fetchTransportData()
        elseif menuItem == "accessories" then fetchAccessoryData()
        elseif menuItem == "houses"      then fetchHouseData()
        elseif menuItem == "skins"       then fetchSkinData()
        end
    end
    imgui.SameLine()
    if imgui.Button("Искать") then
        if     menuItem == "transport"   then fetchTransportData()
        elseif menuItem == "accessories" then fetchAccessoryData()
        elseif menuItem == "houses"      then fetchHouseData()
        elseif menuItem == "skins"       then fetchSkinData()
        end
    end
    imgui.PopItemWidth()
    imgui.Separator()

    imgui.BeginChild("ContentPane", imgui.ImVec2(0, 0), true)

    if loading then
        imgui.Text("Загрузка...")
    elseif menuItem == "transport" then
        if #transportData == 0 then
            imgui.Text("Нет данных.")
        else
            local fullText = "ID\tНазвание\tКласс\tСкорость\tПривод\n"
            for i = 1, #transportData do
                local v = transportData[i]
                fullText = fullText .. string.format(
                    "%s\t%s\t%s\t%s\t%s\n",
                    v.ID or "-", v.Name or "-",
                    (vehicleDetails[v.ID] or {}).Class or "...",
                    (vehicleDetails[v.ID] or {}).Speed or "...",
                    (vehicleDetails[v.ID] or {}).Drive or "..."
                )
                if #fullText > maxOutputLength then
                    fullText = fullText .. "[обрезано...]\n"
                    break
                end
            end
            local bufSize = math.min(#fullText + 128, maxOutputLength + 512)
            local buffer = imgui.ImBuffer(fullText, bufSize)
            imgui.InputTextMultiline("##output", buffer, imgui.ImVec2(-1, -1), imgui.InputTextFlags.ReadOnly)
        end
    elseif menuItem == "accessories" then
        if #accessoryData == 0 then
            imgui.Text("Нет данных.")
        else
            local fullText = "ID\tНазвание\tЦена\n"
            for i = 1, #accessoryData do
                local a = accessoryData[i]
                fullText = fullText .. string.format(
                    "%s\t%s\t%s\n",
                    a.ID or "-", a.Alternative or "-", a.Price or "-"
                )
                if #fullText > maxOutputLength then
                    fullText = fullText .. "[обрезано...]\n"
                    break
                end
            end
            local bufSize = math.min(#fullText + 128, maxOutputLength + 512)
            local buffer = imgui.ImBuffer(fullText, bufSize)
            imgui.InputTextMultiline("##acc_output", buffer, imgui.ImVec2(-1, -1), imgui.InputTextFlags.ReadOnly)
        end
    elseif menuItem == "houses" then
        if not houseData.Class then
            imgui.Text("Введите номер дома и нажмите «Искать».")
        else
            imgui.Text(string.format("Класс:   %s", houseData.Class))
            imgui.Text(string.format("Зона:    %s", houseData.Zone))
            imgui.Text(string.format("Гараж:   %s", houseData.Garage))
        end
    elseif menuItem == "skins" then
        if not skinData or not skinData.Name then
            imgui.Text("Введите ID скина и нажмите «Искать».")
        else
            imgui.Text(string.format("Name:   %s", skinData.Name))
            imgui.Text(string.format("Price:  %s", skinData.Price))
            imgui.Text(string.format("Model:  %s", skinData.Model))
            imgui.Text(string.format("Gender: %s", skinData.Gender))
        end
    else
        imgui.Text("Выберите раздел слева.")
    end

    imgui.EndChild()
    if loading then imgui.EndDisabled() end
    imgui.End()
end


